<?php
/**
 * Copy Trading Pro - User Class
 */

require_once __DIR__ . '/../config/config.php';

class User {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function register($email, $password, $fullName, $phone = null, $country = null) {
        try {
            // Verificar si el email ya existe
            if ($this->emailExists($email)) {
                return ['success' => false, 'message' => 'El email ya está registrado'];
            }
            
            // Validar email
            if (!isValidEmail($email)) {
                return ['success' => false, 'message' => 'Email inválido'];
            }
            
            // Validar password
            if (strlen($password) < 6) {
                return ['success' => false, 'message' => 'La contraseña debe tener al menos 6 caracteres'];
            }
            
            // Hash del password
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            
            // Insertar usuario
            $userId = $this->db->insert('users', [
                'email' => $email,
                'password' => $hashedPassword,
                'full_name' => $fullName,
                'phone' => $phone,
                'country' => $country,
                'role' => 'follower',
                'status' => 'active'
            ]);
            
            if ($userId) {
                logMessage('info', 'New user registered', ['user_id' => $userId, 'email' => $email]);
                
                // Crear notificación de bienvenida
                $this->createNotification($userId, 'system', 'Bienvenido a Copy Trading Pro', 
                    'Gracias por registrarte. Empieza conectando tu cuenta MT4/MT5.');
                
                return ['success' => true, 'message' => 'Usuario registrado exitosamente', 'user_id' => $userId];
            }
            
            return ['success' => false, 'message' => 'Error al registrar usuario'];
            
        } catch (Exception $e) {
            logMessage('error', 'Registration error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error al registrar usuario'];
        }
    }
    
    public function login($email, $password) {
        try {
            $user = $this->db->fetchOne(
                "SELECT * FROM users WHERE email = ? AND status = 'active'",
                [$email]
            );
            
            if (!$user) {
                logMessage('warning', 'Login attempt with invalid email', ['email' => $email]);
                return ['success' => false, 'message' => 'Credenciales inválidas'];
            }
            
            if (!password_verify($password, $user['password'])) {
                logMessage('warning', 'Login attempt with invalid password', ['user_id' => $user['id']]);
                return ['success' => false, 'message' => 'Credenciales inválidas'];
            }
            
            // Actualizar último login
            $this->db->update('users', 
                ['last_login' => date('Y-m-d H:i:s')],
                'id = ?',
                [$user['id']]
            );
            
            // Remover password del array
            unset($user['password']);
            
            // Guardar en sesión
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user'] = $user;
            $_SESSION['logged_in'] = true;
            
            logMessage('info', 'User logged in', ['user_id' => $user['id']]);
            
            return ['success' => true, 'message' => 'Login exitoso', 'user' => $user];
            
        } catch (Exception $e) {
            logMessage('error', 'Login error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error al iniciar sesión'];
        }
    }
    
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            logMessage('info', 'User logged out', ['user_id' => $_SESSION['user_id']]);
        }
        
        session_unset();
        session_destroy();
        
        return ['success' => true, 'message' => 'Sesión cerrada'];
    }
    
    public function getUserById($userId) {
        return $this->db->fetchOne(
            "SELECT id, email, full_name, role, status, phone, country, avatar, created_at, last_login 
             FROM users WHERE id = ?",
            [$userId]
        );
    }
    
    public function getAllUsers($filters = []) {
        $sql = "SELECT id, email, full_name, role, status, phone, country, created_at, last_login 
                FROM users WHERE 1=1";
        $params = [];
        
        if (isset($filters['role'])) {
            $sql .= " AND role = ?";
            $params[] = $filters['role'];
        }
        
        if (isset($filters['status'])) {
            $sql .= " AND status = ?";
            $params[] = $filters['status'];
        }
        
        if (isset($filters['search'])) {
            $sql .= " AND (email LIKE ? OR full_name LIKE ?)";
            $search = '%' . $filters['search'] . '%';
            $params[] = $search;
            $params[] = $search;
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        if (isset($filters['limit'])) {
            $sql .= " LIMIT ?";
            $params[] = (int)$filters['limit'];
        }
        
        return $this->db->fetchAll($sql, $params);
    }
    
    public function updateUser($userId, $data) {
        try {
            // Si se está actualizando el password, hashearlo
            if (isset($data['password'])) {
                if (strlen($data['password']) < 6) {
                    return ['success' => false, 'message' => 'La contraseña debe tener al menos 6 caracteres'];
                }
                $data['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
            }
            
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            $affected = $this->db->update('users', $data, 'id = ?', [$userId]);
            
            if ($affected > 0) {
                logMessage('info', 'User updated', ['user_id' => $userId]);
                return ['success' => true, 'message' => 'Usuario actualizado exitosamente'];
            }
            
            return ['success' => false, 'message' => 'No se realizaron cambios'];
            
        } catch (Exception $e) {
            logMessage('error', 'User update error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error al actualizar usuario'];
        }
    }
    
    public function deleteUser($userId) {
        try {
            $affected = $this->db->delete('users', 'id = ?', [$userId]);
            
            if ($affected > 0) {
                logMessage('info', 'User deleted', ['user_id' => $userId]);
                return ['success' => true, 'message' => 'Usuario eliminado exitosamente'];
            }
            
            return ['success' => false, 'message' => 'Usuario no encontrado'];
            
        } catch (Exception $e) {
            logMessage('error', 'User delete error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error al eliminar usuario'];
        }
    }
    
    public function changePassword($userId, $currentPassword, $newPassword) {
        try {
            $user = $this->db->fetchOne(
                "SELECT password FROM users WHERE id = ?",
                [$userId]
            );
            
            if (!$user) {
                return ['success' => false, 'message' => 'Usuario no encontrado'];
            }
            
            if (!password_verify($currentPassword, $user['password'])) {
                return ['success' => false, 'message' => 'Contraseña actual incorrecta'];
            }
            
            if (strlen($newPassword) < 6) {
                return ['success' => false, 'message' => 'La nueva contraseña debe tener al menos 6 caracteres'];
            }
            
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            
            $this->db->update('users',
                ['password' => $hashedPassword],
                'id = ?',
                [$userId]
            );
            
            logMessage('info', 'Password changed', ['user_id' => $userId]);
            
            return ['success' => true, 'message' => 'Contraseña cambiada exitosamente'];
            
        } catch (Exception $e) {
            logMessage('error', 'Password change error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error al cambiar contraseña'];
        }
    }
    
    public function emailExists($email) {
        $count = $this->db->fetchColumn(
            "SELECT COUNT(*) FROM users WHERE email = ?",
            [$email]
        );
        return $count > 0;
    }
    
    public function getUserStats($userId) {
        return $this->db->fetchOne(
            "SELECT * FROM user_stats WHERE id = ?",
            [$userId]
        );
    }
    
    public function createNotification($userId, $type, $title, $message, $link = null) {
        try {
            $this->db->insert('notifications', [
                'user_id' => $userId,
                'type' => $type,
                'title' => $title,
                'message' => $message,
                'link' => $link
            ]);
            return true;
        } catch (Exception $e) {
            logMessage('error', 'Notification creation error: ' . $e->getMessage());
            return false;
        }
    }
    
    public function getNotifications($userId, $limit = 10, $unreadOnly = false) {
        $sql = "SELECT * FROM notifications WHERE user_id = ?";
        $params = [$userId];
        
        if ($unreadOnly) {
            $sql .= " AND is_read = 0";
        }
        
        $sql .= " ORDER BY created_at DESC LIMIT ?";
        $params[] = $limit;
        
        return $this->db->fetchAll($sql, $params);
    }
    
    public function markNotificationAsRead($notificationId) {
        return $this->db->update('notifications',
            ['is_read' => 1],
            'id = ?',
            [$notificationId]
        );
    }
    
    public function getUnreadNotificationCount($userId) {
        return $this->db->fetchColumn(
            "SELECT COUNT(*) FROM notifications WHERE user_id = ? AND is_read = 0",
            [$userId]
        );
    }
}
