<?php
require_once __DIR__ . '/../config/config.php';

class Subscription {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function createSubscription($userId, $planId, $paymentMethod = 'manual') {
        $plan = $this->getPlan($planId);
        if (!$plan) {
            return ['success' => false, 'message' => 'Plan not found'];
        }
        
        $startDate = date('Y-m-d');
        $endDate = date('Y-m-d', strtotime('+1 ' . $plan['billing_cycle']));
        
        try {
            $subscriptionId = $this->db->insert('subscriptions', [
                'user_id' => $userId,
                'plan_id' => $planId,
                'status' => 'active',
                'start_date' => $startDate,
                'end_date' => $endDate,
                'auto_renew' => true
            ]);
            
            $this->db->insert('transactions', [
                'user_id' => $userId,
                'subscription_id' => $subscriptionId,
                'amount' => $plan['price'],
                'currency' => 'USD',
                'payment_method' => $paymentMethod,
                'status' => 'completed',
                'description' => 'Subscription to ' . $plan['name']
            ]);
            
            logMessage('info', 'Subscription created', [
                'user_id' => $userId,
                'subscription_id' => $subscriptionId
            ]);
            
            return ['success' => true, 'subscription_id' => $subscriptionId];
            
        } catch (Exception $e) {
            logMessage('error', 'Subscription creation failed: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error creating subscription'];
        }
    }
    
    public function getPlan($planId) {
        return $this->db->fetchOne(
            "SELECT * FROM subscription_plans WHERE id = ? AND is_active = 1",
            [$planId]
        );
    }
    
    public function getAllPlans() {
        return $this->db->fetchAll(
            "SELECT * FROM subscription_plans WHERE is_active = 1 ORDER BY price ASC"
        );
    }
    
    public function getUserSubscription($userId) {
        return $this->db->fetchOne(
            "SELECT s.*, p.name as plan_name, p.max_accounts, p.max_copy_traders
             FROM subscriptions s
             JOIN subscription_plans p ON s.plan_id = p.id
             WHERE s.user_id = ? AND s.status = 'active'
             ORDER BY s.end_date DESC LIMIT 1",
            [$userId]
        );
    }
}
