<?php
/**
 * Copy Trading Pro - MetaTrader API Integration
 */

require_once __DIR__ . '/../config/config.php';

class MetaTraderAPI {
    private $apiUrl;
    private $apiToken;
    private $db;
    
    public function __construct() {
        $this->apiUrl = METAAPI_URL;
        $this->apiToken = METAAPI_TOKEN;
        $this->db = Database::getInstance();
    }
    
    private function makeRequest($endpoint, $method = 'GET', $data = null) {
        $url = $this->apiUrl . $endpoint;
        
        $headers = [
            'Content-Type: application/json',
            'auth-token: ' . $this->apiToken
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'PUT') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return json_decode($response, true);
        }
        
        logMessage('error', 'MetaAPI request failed', [
            'endpoint' => $endpoint,
            'method' => $method,
            'http_code' => $httpCode,
            'response' => $response
        ]);
        
        return null;
    }
    
    public function connectAccount($accountNumber, $broker, $server, $password, $platform = 'mt4') {
        $data = [
            'login' => $accountNumber,
            'password' => $password,
            'name' => $accountNumber . '@' . $broker,
            'server' => $server,
            'platform' => $platform,
            'magic' => 0,
            'application' => 'CopyTradingPro'
        ];
        
        return $this->makeRequest('/users/current/accounts', 'POST', $data);
    }
    
    public function getAccountInfo($accountId) {
        return $this->makeRequest('/users/current/accounts/' . $accountId);
    }
    
    public function getAccountBalance($accountId) {
        $info = $this->makeRequest('/users/current/accounts/' . $accountId . '/account-information');
        return $info;
    }
    
    public function getOpenPositions($accountId) {
        return $this->makeRequest('/users/current/accounts/' . $accountId . '/positions');
    }
    
    public function getOrders($accountId) {
        return $this->makeRequest('/users/current/accounts/' . $accountId . '/orders');
    }
    
    public function getHistory($accountId, $startTime = null, $endTime = null) {
        $endpoint = '/users/current/accounts/' . $accountId . '/history-orders/time';
        
        if ($startTime) {
            $endpoint .= '/' . $startTime;
            if ($endTime) {
                $endpoint .= '/' . $endTime;
            }
        }
        
        return $this->makeRequest($endpoint);
    }
    
    public function placeTrade($accountId, $symbol, $type, $volume, $stopLoss = null, $takeProfit = null, $comment = null) {
        $data = [
            'actionType' => 'ORDER_TYPE_' . strtoupper($type),
            'symbol' => $symbol,
            'volume' => $volume
        ];
        
        if ($stopLoss) {
            $data['stopLoss'] = $stopLoss;
        }
        
        if ($takeProfit) {
            $data['takeProfit'] = $takeProfit;
        }
        
        if ($comment) {
            $data['comment'] = $comment;
        }
        
        return $this->makeRequest('/users/current/accounts/' . $accountId . '/trade', 'POST', $data);
    }
    
    public function closeTrade($accountId, $positionId) {
        return $this->makeRequest(
            '/users/current/accounts/' . $accountId . '/positions/' . $positionId . '/close',
            'POST'
        );
    }
    
    public function modifyPosition($accountId, $positionId, $stopLoss = null, $takeProfit = null) {
        $data = [];
        
        if ($stopLoss !== null) {
            $data['stopLoss'] = $stopLoss;
        }
        
        if ($takeProfit !== null) {
            $data['takeProfit'] = $takeProfit;
        }
        
        return $this->makeRequest(
            '/users/current/accounts/' . $accountId . '/positions/' . $positionId,
            'PUT',
            $data
        );
    }
    
    public function syncAccount($accountId) {
        try {
            $accountInfo = $this->getAccountBalance($accountId);
            
            if ($accountInfo) {
                $this->db->update('mt_accounts', [
                    'balance' => $accountInfo['balance'] ?? 0,
                    'equity' => $accountInfo['equity'] ?? 0,
                    'margin' => $accountInfo['margin'] ?? 0,
                    'free_margin' => $accountInfo['freeMargin'] ?? 0,
                    'last_sync' => date('Y-m-d H:i:s')
                ], 'id = ?', [$accountId]);
                
                return true;
            }
            
            return false;
            
        } catch (Exception $e) {
            logMessage('error', 'Account sync error: ' . $e->getMessage());
            return false;
        }
    }
}
