<?php
require_once __DIR__ . '/../config/config.php';

class CopyTrading {
    private $db;
    private $mtApi;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->mtApi = new MetaTraderAPI();
    }
    
    public function createCopySettings($followerAccountId, $masterAccountId, $settings = []) {
        $defaultSettings = [
            'is_active' => true,
            'lot_multiplier' => 1.00,
            'reverse_copy' => false,
            'max_lot_size' => 10.00,
            'stop_loss_pips' => 0,
            'take_profit_pips' => 0,
            'max_slippage' => 5,
            'copy_pending_orders' => true
        ];
        
        $settings = array_merge($defaultSettings, $settings);
        $settings['follower_account_id'] = $followerAccountId;
        $settings['master_account_id'] = $masterAccountId;
        
        try {
            $id = $this->db->insert('copy_settings', $settings);
            logMessage('info', 'Copy settings created', ['id' => $id]);
            return ['success' => true, 'id' => $id];
        } catch (Exception $e) {
            logMessage('error', 'Copy settings creation failed: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error creating copy settings'];
        }
    }
    
    public function syncTrades() {
        $activeSettings = $this->db->fetchAll(
            "SELECT cs.*, ma_master.api_token as master_token, ma_follower.api_token as follower_token
             FROM copy_settings cs
             JOIN mt_accounts ma_master ON cs.master_account_id = ma_master.id
             JOIN mt_accounts ma_follower ON cs.follower_account_id = ma_follower.id
             WHERE cs.is_active = 1 AND ma_master.is_active = 1 AND ma_follower.is_active = 1"
        );
        
        foreach ($activeSettings as $setting) {
            $this->copyTradesForSetting($setting);
        }
    }
    
    private function copyTradesForSetting($setting) {
        try {
            $masterPositions = $this->mtApi->getOpenPositions($setting['master_account_id']);
            
            if (!$masterPositions) return;
            
            foreach ($masterPositions as $position) {
                $existingCopy = $this->db->fetchOne(
                    "SELECT * FROM copied_trades WHERE copy_setting_id = ? AND master_ticket = ? AND status = 'open'",
                    [$setting['id'], $position['id']]
                );
                
                if (!$existingCopy) {
                    $this->copyTrade($setting, $position);
                }
            }
            
        } catch (Exception $e) {
            logMessage('error', 'Trade sync error: ' . $e->getMessage());
        }
    }
    
    private function copyTrade($setting, $masterPosition) {
        $volume = $masterPosition['volume'] * $setting['lot_multiplier'];
        
        if ($volume > $setting['max_lot_size']) {
            $volume = $setting['max_lot_size'];
        }
        
        $type = $setting['reverse_copy'] ? 
            ($masterPosition['type'] === 'buy' ? 'sell' : 'buy') : 
            $masterPosition['type'];
        
        $stopLoss = $masterPosition['stopLoss'];
        $takeProfit = $masterPosition['takeProfit'];
        
        $result = $this->mtApi->placeTrade(
            $setting['follower_account_id'],
            $masterPosition['symbol'],
            $type,
            $volume,
            $stopLoss,
            $takeProfit,
            'Copied from ' . $setting['master_account_id']
        );
        
        if ($result && isset($result['positionId'])) {
            $this->db->insert('copied_trades', [
                'copy_setting_id' => $setting['id'],
                'master_ticket' => $masterPosition['id'],
                'follower_ticket' => $result['positionId'],
                'symbol' => $masterPosition['symbol'],
                'trade_type' => $type,
                'lot_size' => $volume,
                'open_price' => $masterPosition['openPrice'],
                'stop_loss' => $stopLoss,
                'take_profit' => $takeProfit,
                'status' => 'open',
                'open_time' => date('Y-m-d H:i:s')
            ]);
            
            logMessage('info', 'Trade copied successfully', [
                'master_ticket' => $masterPosition['id'],
                'follower_ticket' => $result['positionId']
            ]);
        }
    }
}
