-- =====================================================
-- COPY TRADING PRO - DATABASE SCHEMA
-- =====================================================

-- Eliminar tablas si existen
DROP TABLE IF EXISTS notifications;
DROP TABLE IF EXISTS statistics;
DROP TABLE IF EXISTS copied_trades;
DROP TABLE IF EXISTS copy_settings;
DROP TABLE IF EXISTS mt_accounts;
DROP TABLE IF EXISTS transactions;
DROP TABLE IF EXISTS subscriptions;
DROP TABLE IF EXISTS subscription_plans;
DROP TABLE IF EXISTS users;
DROP TABLE IF EXISTS system_logs;

-- =====================================================
-- TABLA: users
-- =====================================================
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    full_name VARCHAR(255) NOT NULL,
    role ENUM('admin', 'trader', 'follower') DEFAULT 'follower',
    status ENUM('active', 'suspended', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL,
    phone VARCHAR(50),
    country VARCHAR(100),
    avatar VARCHAR(255),
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: subscription_plans
-- =====================================================
CREATE TABLE subscription_plans (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    price DECIMAL(10,2) NOT NULL,
    billing_cycle ENUM('monthly', 'quarterly', 'yearly') DEFAULT 'monthly',
    max_accounts INT DEFAULT 1,
    max_copy_traders INT DEFAULT 1,
    features TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: subscriptions
-- =====================================================
CREATE TABLE subscriptions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    plan_id INT NOT NULL,
    status ENUM('active', 'expired', 'cancelled', 'pending') DEFAULT 'pending',
    start_date DATE NOT NULL,
    end_date DATE NOT NULL,
    auto_renew BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (plan_id) REFERENCES subscription_plans(id),
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_dates (start_date, end_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: transactions
-- =====================================================
CREATE TABLE transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    subscription_id INT,
    amount DECIMAL(10,2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'USD',
    payment_method ENUM('stripe', 'paypal', 'manual') NOT NULL,
    transaction_id VARCHAR(255),
    status ENUM('completed', 'pending', 'failed', 'refunded') DEFAULT 'pending',
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (subscription_id) REFERENCES subscriptions(id),
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_date (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: mt_accounts
-- =====================================================
CREATE TABLE mt_accounts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    account_type ENUM('mt4', 'mt5') NOT NULL,
    account_number VARCHAR(50) NOT NULL,
    broker VARCHAR(100) NOT NULL,
    server VARCHAR(100),
    password VARCHAR(255) NOT NULL,
    is_master BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    balance DECIMAL(15,2) DEFAULT 0,
    equity DECIMAL(15,2) DEFAULT 0,
    margin DECIMAL(15,2) DEFAULT 0,
    free_margin DECIMAL(15,2) DEFAULT 0,
    leverage INT DEFAULT 100,
    api_token VARCHAR(255),
    last_sync TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user (user_id),
    INDEX idx_master (is_master),
    INDEX idx_active (is_active),
    UNIQUE KEY unique_account (account_number, broker)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: copy_settings
-- =====================================================
CREATE TABLE copy_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    follower_account_id INT NOT NULL,
    master_account_id INT NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    lot_multiplier DECIMAL(5,2) DEFAULT 1.00,
    reverse_copy BOOLEAN DEFAULT FALSE,
    max_lot_size DECIMAL(10,2) DEFAULT 10.00,
    stop_loss_pips INT DEFAULT 0,
    take_profit_pips INT DEFAULT 0,
    copy_symbols TEXT,
    skip_symbols TEXT,
    max_slippage INT DEFAULT 5,
    copy_pending_orders BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (follower_account_id) REFERENCES mt_accounts(id) ON DELETE CASCADE,
    FOREIGN KEY (master_account_id) REFERENCES mt_accounts(id) ON DELETE CASCADE,
    INDEX idx_follower (follower_account_id),
    INDEX idx_master (master_account_id),
    INDEX idx_active (is_active),
    UNIQUE KEY unique_copy (follower_account_id, master_account_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: copied_trades
-- =====================================================
CREATE TABLE copied_trades (
    id INT AUTO_INCREMENT PRIMARY KEY,
    copy_setting_id INT NOT NULL,
    master_ticket BIGINT NOT NULL,
    follower_ticket BIGINT,
    symbol VARCHAR(20) NOT NULL,
    trade_type ENUM('buy', 'sell') NOT NULL,
    lot_size DECIMAL(10,2) NOT NULL,
    open_price DECIMAL(15,5) NOT NULL,
    close_price DECIMAL(15,5),
    stop_loss DECIMAL(15,5),
    take_profit DECIMAL(15,5),
    profit DECIMAL(15,2),
    commission DECIMAL(10,2) DEFAULT 0,
    swap DECIMAL(10,2) DEFAULT 0,
    status ENUM('open', 'closed', 'failed') DEFAULT 'open',
    open_time TIMESTAMP NOT NULL,
    close_time TIMESTAMP NULL,
    comment TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (copy_setting_id) REFERENCES copy_settings(id) ON DELETE CASCADE,
    INDEX idx_copy_setting (copy_setting_id),
    INDEX idx_master_ticket (master_ticket),
    INDEX idx_status (status),
    INDEX idx_symbol (symbol),
    INDEX idx_open_time (open_time)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: statistics
-- =====================================================
CREATE TABLE statistics (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_id INT NOT NULL,
    stat_date DATE NOT NULL,
    total_trades INT DEFAULT 0,
    winning_trades INT DEFAULT 0,
    losing_trades INT DEFAULT 0,
    total_profit DECIMAL(15,2) DEFAULT 0,
    total_loss DECIMAL(15,2) DEFAULT 0,
    net_profit DECIMAL(15,2) DEFAULT 0,
    win_rate DECIMAL(5,2) DEFAULT 0,
    profit_factor DECIMAL(10,2) DEFAULT 0,
    max_drawdown DECIMAL(15,2) DEFAULT 0,
    average_win DECIMAL(15,2) DEFAULT 0,
    average_loss DECIMAL(15,2) DEFAULT 0,
    largest_win DECIMAL(15,2) DEFAULT 0,
    largest_loss DECIMAL(15,2) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (account_id) REFERENCES mt_accounts(id) ON DELETE CASCADE,
    INDEX idx_account (account_id),
    INDEX idx_date (stat_date),
    UNIQUE KEY unique_stat (account_id, stat_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: notifications
-- =====================================================
CREATE TABLE notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    type ENUM('trade', 'subscription', 'system', 'alert') DEFAULT 'system',
    title VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    is_read BOOLEAN DEFAULT FALSE,
    link VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user (user_id),
    INDEX idx_read (is_read),
    INDEX idx_type (type),
    INDEX idx_date (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TABLA: system_logs
-- =====================================================
CREATE TABLE system_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    action VARCHAR(100) NOT NULL,
    description TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_action (action),
    INDEX idx_date (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- DATOS DE EJEMPLO
-- =====================================================

-- Usuario administrador
INSERT INTO users (email, password, full_name, role, status) VALUES 
('admin@copytrading.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Administrador', 'admin', 'active');
-- Password: password

-- Planes de suscripción
INSERT INTO subscription_plans (name, description, price, billing_cycle, max_accounts, max_copy_traders, features, is_active) VALUES 
('Basic', 'Plan básico para principiantes', 29.99, 'monthly', 1, 1, '["1 cuenta MT4/MT5", "Copiar 1 trader", "Soporte por email"]', TRUE),
('Pro', 'Plan profesional con más características', 79.99, 'monthly', 3, 5, '["3 cuentas MT4/MT5", "Copiar hasta 5 traders", "Soporte prioritario", "Estadísticas avanzadas"]', TRUE),
('Premium', 'Plan premium sin limitaciones', 199.99, 'monthly', 10, 20, '["10 cuentas MT4/MT5", "Copiar hasta 20 traders", "Soporte VIP 24/7", "Estadísticas avanzadas", "API access", "Risk management tools"]', TRUE),
('Enterprise', 'Solución empresarial personalizada', 499.99, 'monthly', -1, -1, '["Cuentas ilimitadas", "Traders ilimitados", "Soporte dedicado", "White label", "API completo", "Custom features"]', TRUE);

-- =====================================================
-- VISTAS ÚTILES
-- =====================================================

-- Vista de estadísticas de usuarios
CREATE OR REPLACE VIEW user_stats AS
SELECT 
    u.id,
    u.full_name,
    u.email,
    u.role,
    COUNT(DISTINCT ma.id) as total_accounts,
    COUNT(DISTINCT CASE WHEN ma.is_master = TRUE THEN ma.id END) as master_accounts,
    COUNT(DISTINCT cs.id) as active_copies,
    COALESCE(SUM(s.net_profit), 0) as total_profit
FROM users u
LEFT JOIN mt_accounts ma ON u.id = ma.user_id
LEFT JOIN copy_settings cs ON ma.id = cs.follower_account_id AND cs.is_active = TRUE
LEFT JOIN statistics s ON ma.id = s.account_id
GROUP BY u.id;

-- Vista de performance de traders master
CREATE OR REPLACE VIEW master_trader_performance AS
SELECT 
    ma.id as account_id,
    u.id as user_id,
    u.full_name,
    ma.account_number,
    ma.broker,
    COUNT(DISTINCT cs.id) as followers_count,
    COALESCE(AVG(s.win_rate), 0) as avg_win_rate,
    COALESCE(SUM(s.net_profit), 0) as total_profit,
    COALESCE(AVG(s.profit_factor), 0) as avg_profit_factor,
    MAX(s.stat_date) as last_stat_date
FROM mt_accounts ma
INNER JOIN users u ON ma.user_id = u.id
LEFT JOIN copy_settings cs ON ma.id = cs.master_account_id AND cs.is_active = TRUE
LEFT JOIN statistics s ON ma.id = s.account_id
WHERE ma.is_master = TRUE AND ma.is_active = TRUE
GROUP BY ma.id
ORDER BY followers_count DESC, total_profit DESC;
