#!/bin/bash

# Crear archivos PHP adicionales

# CopyTrading.php
cat > src/CopyTrading.php << 'PHPCODE'
<?php
require_once __DIR__ . '/../config/config.php';

class CopyTrading {
    private $db;
    private $mtApi;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->mtApi = new MetaTraderAPI();
    }
    
    public function createCopySettings($followerAccountId, $masterAccountId, $settings = []) {
        $defaultSettings = [
            'is_active' => true,
            'lot_multiplier' => 1.00,
            'reverse_copy' => false,
            'max_lot_size' => 10.00,
            'stop_loss_pips' => 0,
            'take_profit_pips' => 0,
            'max_slippage' => 5,
            'copy_pending_orders' => true
        ];
        
        $settings = array_merge($defaultSettings, $settings);
        $settings['follower_account_id'] = $followerAccountId;
        $settings['master_account_id'] = $masterAccountId;
        
        try {
            $id = $this->db->insert('copy_settings', $settings);
            logMessage('info', 'Copy settings created', ['id' => $id]);
            return ['success' => true, 'id' => $id];
        } catch (Exception $e) {
            logMessage('error', 'Copy settings creation failed: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error creating copy settings'];
        }
    }
    
    public function syncTrades() {
        $activeSettings = $this->db->fetchAll(
            "SELECT cs.*, ma_master.api_token as master_token, ma_follower.api_token as follower_token
             FROM copy_settings cs
             JOIN mt_accounts ma_master ON cs.master_account_id = ma_master.id
             JOIN mt_accounts ma_follower ON cs.follower_account_id = ma_follower.id
             WHERE cs.is_active = 1 AND ma_master.is_active = 1 AND ma_follower.is_active = 1"
        );
        
        foreach ($activeSettings as $setting) {
            $this->copyTradesForSetting($setting);
        }
    }
    
    private function copyTradesForSetting($setting) {
        try {
            $masterPositions = $this->mtApi->getOpenPositions($setting['master_account_id']);
            
            if (!$masterPositions) return;
            
            foreach ($masterPositions as $position) {
                $existingCopy = $this->db->fetchOne(
                    "SELECT * FROM copied_trades WHERE copy_setting_id = ? AND master_ticket = ? AND status = 'open'",
                    [$setting['id'], $position['id']]
                );
                
                if (!$existingCopy) {
                    $this->copyTrade($setting, $position);
                }
            }
            
        } catch (Exception $e) {
            logMessage('error', 'Trade sync error: ' . $e->getMessage());
        }
    }
    
    private function copyTrade($setting, $masterPosition) {
        $volume = $masterPosition['volume'] * $setting['lot_multiplier'];
        
        if ($volume > $setting['max_lot_size']) {
            $volume = $setting['max_lot_size'];
        }
        
        $type = $setting['reverse_copy'] ? 
            ($masterPosition['type'] === 'buy' ? 'sell' : 'buy') : 
            $masterPosition['type'];
        
        $stopLoss = $masterPosition['stopLoss'];
        $takeProfit = $masterPosition['takeProfit'];
        
        $result = $this->mtApi->placeTrade(
            $setting['follower_account_id'],
            $masterPosition['symbol'],
            $type,
            $volume,
            $stopLoss,
            $takeProfit,
            'Copied from ' . $setting['master_account_id']
        );
        
        if ($result && isset($result['positionId'])) {
            $this->db->insert('copied_trades', [
                'copy_setting_id' => $setting['id'],
                'master_ticket' => $masterPosition['id'],
                'follower_ticket' => $result['positionId'],
                'symbol' => $masterPosition['symbol'],
                'trade_type' => $type,
                'lot_size' => $volume,
                'open_price' => $masterPosition['openPrice'],
                'stop_loss' => $stopLoss,
                'take_profit' => $takeProfit,
                'status' => 'open',
                'open_time' => date('Y-m-d H:i:s')
            ]);
            
            logMessage('info', 'Trade copied successfully', [
                'master_ticket' => $masterPosition['id'],
                'follower_ticket' => $result['positionId']
            ]);
        }
    }
}
PHPCODE

# Subscription.php
cat > src/Subscription.php << 'PHPCODE'
<?php
require_once __DIR__ . '/../config/config.php';

class Subscription {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function createSubscription($userId, $planId, $paymentMethod = 'manual') {
        $plan = $this->getPlan($planId);
        if (!$plan) {
            return ['success' => false, 'message' => 'Plan not found'];
        }
        
        $startDate = date('Y-m-d');
        $endDate = date('Y-m-d', strtotime('+1 ' . $plan['billing_cycle']));
        
        try {
            $subscriptionId = $this->db->insert('subscriptions', [
                'user_id' => $userId,
                'plan_id' => $planId,
                'status' => 'active',
                'start_date' => $startDate,
                'end_date' => $endDate,
                'auto_renew' => true
            ]);
            
            $this->db->insert('transactions', [
                'user_id' => $userId,
                'subscription_id' => $subscriptionId,
                'amount' => $plan['price'],
                'currency' => 'USD',
                'payment_method' => $paymentMethod,
                'status' => 'completed',
                'description' => 'Subscription to ' . $plan['name']
            ]);
            
            logMessage('info', 'Subscription created', [
                'user_id' => $userId,
                'subscription_id' => $subscriptionId
            ]);
            
            return ['success' => true, 'subscription_id' => $subscriptionId];
            
        } catch (Exception $e) {
            logMessage('error', 'Subscription creation failed: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Error creating subscription'];
        }
    }
    
    public function getPlan($planId) {
        return $this->db->fetchOne(
            "SELECT * FROM subscription_plans WHERE id = ? AND is_active = 1",
            [$planId]
        );
    }
    
    public function getAllPlans() {
        return $this->db->fetchAll(
            "SELECT * FROM subscription_plans WHERE is_active = 1 ORDER BY price ASC"
        );
    }
    
    public function getUserSubscription($userId) {
        return $this->db->fetchOne(
            "SELECT s.*, p.name as plan_name, p.max_accounts, p.max_copy_traders
             FROM subscriptions s
             JOIN subscription_plans p ON s.plan_id = p.id
             WHERE s.user_id = ? AND s.status = 'active'
             ORDER BY s.end_date DESC LIMIT 1",
            [$userId]
        );
    }
}
PHPCODE

# Statistics.php
cat > src/Statistics.php << 'PHPCODE'
<?php
require_once __DIR__ . '/../config/config.php';

class Statistics {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function generateDailyStats($accountId, $date = null) {
        if (!$date) {
            $date = date('Y-m-d');
        }
        
        $trades = $this->db->fetchAll(
            "SELECT * FROM copied_trades 
             WHERE DATE(close_time) = ? AND status = 'closed'
             AND copy_setting_id IN (
                 SELECT id FROM copy_settings WHERE follower_account_id = ? OR master_account_id = ?
             )",
            [$date, $accountId, $accountId]
        );
        
        $totalTrades = count($trades);
        $winningTrades = 0;
        $losingTrades = 0;
        $totalProfit = 0;
        $totalLoss = 0;
        $largestWin = 0;
        $largestLoss = 0;
        
        foreach ($trades as $trade) {
            if ($trade['profit'] > 0) {
                $winningTrades++;
                $totalProfit += $trade['profit'];
                if ($trade['profit'] > $largestWin) {
                    $largestWin = $trade['profit'];
                }
            } else {
                $losingTrades++;
                $totalLoss += abs($trade['profit']);
                if (abs($trade['profit']) > $largestLoss) {
                    $largestLoss = abs($trade['profit']);
                }
            }
        }
        
        $winRate = $totalTrades > 0 ? ($winningTrades / $totalTrades) * 100 : 0;
        $netProfit = $totalProfit - $totalLoss;
        $profitFactor = $totalLoss > 0 ? $totalProfit / $totalLoss : 0;
        $avgWin = $winningTrades > 0 ? $totalProfit / $winningTrades : 0;
        $avgLoss = $losingTrades > 0 ? $totalLoss / $losingTrades : 0;
        
        $this->db->query(
            "INSERT INTO statistics (account_id, stat_date, total_trades, winning_trades, losing_trades,
             total_profit, total_loss, net_profit, win_rate, profit_factor, average_win, average_loss,
             largest_win, largest_loss)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
             ON DUPLICATE KEY UPDATE
             total_trades = VALUES(total_trades),
             winning_trades = VALUES(winning_trades),
             losing_trades = VALUES(losing_trades),
             total_profit = VALUES(total_profit),
             total_loss = VALUES(total_loss),
             net_profit = VALUES(net_profit),
             win_rate = VALUES(win_rate),
             profit_factor = VALUES(profit_factor),
             average_win = VALUES(average_win),
             average_loss = VALUES(average_loss),
             largest_win = VALUES(largest_win),
             largest_loss = VALUES(largest_loss)",
            [
                $accountId, $date, $totalTrades, $winningTrades, $losingTrades,
                $totalProfit, $totalLoss, $netProfit, $winRate, $profitFactor,
                $avgWin, $avgLoss, $largestWin, $largestLoss
            ]
        );
        
        return true;
    }
    
    public function getAccountStats($accountId, $days = 30) {
        return $this->db->fetchAll(
            "SELECT * FROM statistics 
             WHERE account_id = ? AND stat_date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
             ORDER BY stat_date DESC",
            [$accountId, $days]
        );
    }
}
PHPCODE

echo "Core classes created"
